# =============================================================================
#  Analysis 4: This example script demonstrates intra-vital analysis using supplementary movie 1 from Li et al. 2012
#
# [Li et al., 2012]: Li, J. L., Goh, C. C., Keeble, J. L., Qin, J. S., Roediger, B., Jain, R., ... & Ng, L. G. (2012). Intravital multiphoton imaging of immune responses in the mouse ear skin. Nature protocols, 7(2), 221.
#
# =============================================================================

def read_movie(moviefile, resize=1.):
    """
    for more information on using the moviepy library
    https://zulko.github.io/moviepy/
    """
    from moviepy.editor import VideoFileClip, ImageClip
    from tqdm import tqdm 
    from skimage.transform import rescale
    import numpy as np 
    
    vidframes = []
    clip = VideoFileClip(moviefile)
    count = 0
    for frame in tqdm(clip.iter_frames()):
        vidframes.append(np.uint8(rescale(frame, 1./resize, preserve_range=True)))
        count+=1
    
    return np.array(vidframes)

moviefile = '../Data/Videos/nprot.2011.438-S5.mov'
movie = read_movie(moviefile, resize=1.) 

n_frame, n_rows, n_cols, n_channels = movie.shape
print('Size of video: (%d,%d,%d,%d)' %(n_frame,n_rows,n_cols,n_channels))

# get the movement of the green cells. 
"""
1. Motion Extraction
"""
from MOSES.Optical_Flow_Tracking.superpixel_track import compute_grayscale_vid_superpixel_tracks_FB
# motion extraction parameters. 
optical_flow_params = dict(pyr_scale=0.5, levels=5, winsize=21, iterations=5, poly_n=5, poly_sigma=1.2, flags=0)
# number of superpixels
n_spixels = 1000

# extract superpixel tracks for the 2nd or 'green' channel
optflow, meantracks, meantracks_B = compute_grayscale_vid_superpixel_tracks_FB(movie[:,:,:,1], optical_flow_params, n_spixels, dense=True)

# plot the tracks on top of the first frame. 
from MOSES.Visualisation_Tools.track_plotting import plot_tracks
import pylab as plt 
fig, ax = plt.subplots()
ax.imshow(movie[0])
plot_tracks(meantracks, ax, color='g', lw=0.5, alpha=1)
#fig.savefig('green_tracks_cells_intravital-1.svg', dpi=300, bbox_inches='tight')
plt.show()


"""
2. Take a look at the averaged optical flow
"""
from MOSES.Visualisation_Tools.motion_field_visualisation import view_ang_flow
import numpy as np

mean_opt_flow = np.mean(optflow, axis=0)
mean_colored_flow = view_ang_flow(mean_opt_flow)

# create this as an image. 
plt.figure()
plt.imshow(mean_colored_flow)
plt.show()

from skimage.io import imsave
imsave('mean-flow-intravital.tif', mean_colored_flow)

"""
3. Compute the equivalent with MOSES superpixel tracks and visualise the arrow directions. 
"""
from MOSES.Motion_Analysis.tracks_statistics_tools import average_displacement_tracks

mean_disps_tracks =  average_displacement_tracks(meantracks)
# parse out the mean (u,v) velocities of tracks
U_tra = mean_disps_tracks[:,1] 
# this is negative to convert image coordinates to proper (x,y) used in matplotlib quiverplot.
V_tra = -mean_disps_tracks[:,0] 
Mag_tra = np.hypot(U_tra, V_tra)
 
# compute the mean (x,y) position of tracks
X_mean = np.mean(meantracks[:,:,1], axis=-1)
Y_mean = np.mean(meantracks[:,:,0], axis=-1)


# plot the mean velocity tracks onto the first frame 
fig, ax = plt.subplots(figsize=(15,15))
#        fig = plt.figure()
#        fig.set_size_inches(width/height, 1, forward=False)
#        ax = plt.Axes(fig, [0., 0., 1., 1.])
ax.set_axis_off()
#        fig.add_axes(ax)
plt.imshow(movie[0], alpha=0.6)
 
Q = plt.quiver(X_mean[::], Y_mean[::], 
               U_tra[::], V_tra[::], 
               Mag_tra[::], units='xy', pivot='mid', cmap='Greens', scale=None)
plt.show()


"""
4. Motion Saliency Analysis to identify the known 'sink', the site of induced laser injury.
"""
# quantify the degree of localization towards the central square site.
from MOSES.Motion_Analysis.mesh_statistics_tools import compute_motion_saliency_map
from skimage.exposure import equalize_hist
from skimage.filters import gaussian
# specify a large threshold to capture long-distances.
dist_thresh = 5
spixel_size = meantracks[1,0,1]-meantracks[1,0,0]

motion_saliency_F, motion_saliency_spatial_time_F = compute_motion_saliency_map(meantracks, dist_thresh=dist_thresh, 
                                                                  shape=movie.shape[1:-1], max_frame=None, filt=1, filt_size=spixel_size)
motion_saliency_B, motion_saliency_spatial_time_B = compute_motion_saliency_map(meantracks_B, dist_thresh=dist_thresh, 
                                                                  shape=movie.shape[1:-1], max_frame=None, filt=1, filt_size=spixel_size)
# smooth the discrete looking motion saliency maps.
motion_saliency_F_smooth = gaussian(motion_saliency_F, spixel_size/2.)
motion_saliency_B_smooth = gaussian(motion_saliency_B, spixel_size/2.)


fig, ax = plt.subplots(figsize=(15,15))
ax.set_axis_off()
plt.imshow(movie[0], alpha=1)
 
Q = plt.quiver(X_mean[::2], Y_mean[::2], 
               U_tra[::2], V_tra[::2], 
               Mag_tra[::2], units='xy', pivot='mid', cmap='Greens', scale=None)

ax.imshow(motion_saliency_F_smooth/motion_saliency_F_smooth.max(), cmap='Reds', alpha=0.5)
ax.imshow(motion_saliency_B_smooth/motion_saliency_B_smooth.max(), cmap='Greens', alpha=0.5)
#fig.savefig('mean_arrows_and_motion_saliency_maps.svg', dpi=300, bbox_inches='tight')
plt.show()





